%Effective mass accommodation code for outputting the dashed line in Figure
% 3b of Lakey et al. Environ. Sci.: Processes, 2023, 25, 1464-1478

%This code shows the impact of different bulk diffusion coefficients on the
%total film thickness due to partitioning of species with different log Koa
%values after 500 days.

%Species:
%1: SVOC group with a log Koa = 13.5
%2: SVOC group with a log Koa = 12.5
%3: SVOC group with a log Koa = 11.5
%4: SVOC group with a log Koa = 10.5
%5: SVOC group with a log Koa = 9.5
%6: SVOC group with a log Koa = 8.5
warning off all
clear all
close all
global Cgi Koai To p_i
global Db W a0 B_acc Dg T_BL


%----Parameters-----

TSP=2e-11;% total airborne suspended particle conc. (g cm-3, equivalent to 20 micrograms m-3)
To=2e-7;% initial film thickness (cm, equivalent to 2 nm)
fom_TSP=0.4;% volume fraction of organic matter in the particles

p_part=1;% density of airborne particles (g cm-3)
p_i=1;% density of the condensed phase SVOC group i (g cm-3)

Diff_input=logspace(log10(1e-24),log10(1e-6),100); %Range of bulk diffusion coefficients that will be tested (cm2 s-1)

for MM=1:100 %Results for 100 bulk diffusion coefficients will be outputted

    R=8.31e7; % Gas constant (g cm2 s-2 K-1 mol-1)
    T=298; %Temperature (K)
    Db=Diff_input(MM); %Bulk diffusion coefficient (cm2 s-1)
    Dg=0.04; %Gas diffusion coefficient of all species (cm2 s-1)
    T_BL=0.48; %Boundary layer thickness next to the film (cm)
    a0=1; %Surface mass accommodation of all SVOCs on an adsorbate free surface (no units)
    NA=6.02e23; % Avogadro's number (mol-1)

for i=1:6
    Mx(i)=250; % Assumed molar mass of SVOC group i (g mol-1)
    Dens(i)=1; % Density of SVOC group i (g cm-3)
    V_w(i)=Mx(i)/NA/Dens(i); % Volume of one molecule of SVOC group i (cm3)
    W(i)=sqrt(8*R*T/pi/Mx(i)); % Mean thermal velocity of SVOC group i (cm s-1)
end

Cti(1)=0;% total airborne conc. of SVOC group 1 (g cm-3)
Cti(2)=2e-12; % total airborne conc. of SVOC group 2 (g cm-3)
Cti(3)=5e-12; % total airborne conc. of SVOC group 3 (g cm-3)
Cti(4)=10e-12; % total airborne conc. of SVOC group 4 (g cm-3)
Cti(5)=15e-12; % total airborne conc. of SVOC group 5 (g cm-3)
Cti(6)=20e-12; % total airborne conc. of SVOC group 6 (g cm-3)

logKoai(1)=13.5; % log octanal-air partitioning coefficient for SVOC group 1
logKoai(2)=12.5; % log octanal-air partitioning coefficient for SVOC group 2
logKoai(3)=11.5; % log octanal-air partitioning coefficient for SVOC group 3
logKoai(4)=10.5; % log octanal-air partitioning coefficient for SVOC group 4
logKoai(5)=9.5; % log octanal-air partitioning coefficient for SVOC group 5
logKoai(6)=8.5; % log octanal-air partitioning coefficient for SVOC group 6

for i=1:6
    Koai(i)=10^(logKoai(i)); %octanal-air partitioning coefficient for SVOC group i
end

for i=1:6
    Kpi(i)=fom_TSP*Koai(i)/p_part; % particle-air partition coefficient for all SVOC group i (cm3 g-1)
end

for i=1:6
    Cgi(i)=Cti(i)/(TSP*Kpi(i)+1); % gas-phase concentration of SVOC group i (g cm-3)
    Cpi(i)=Cti(i)-Cgi(i); % particle-phase concentration of SVOC group i (g cm-3)
end

for i=1:6
    Length(i)=V_w(i)^(1/3); %Length that a molecule in SVOC group i must travel to be accommodated in the bulk
    B_acc(i)=a0/(1+(Length(i)*W(i)*a0)/(Db*Koai(i)*4)); %Bulk accommodation coefficient
end


%----------Initial concentrations----------------

y0(1)=0; %Initial film mass per film area due to SVOC group 1 (g cm-2)
y0(2)=0; %Initial film mass per film area due to SVOC group 2 (g cm-2)
y0(3)=0; %Initial film mass per film area due to SVOC group 3 (g cm-2)
y0(4)=0; %Initial film mass per film area due to SVOC group 4 (g cm-2)
y0(5)=0; %Initial film mass per film area due to SVOC group 5 (g cm-2)
y0(6)=0; %Initial film mass per film area due to SVOC group 6 (g cm-2)


%-----------Solve differential equations----------------------
n=9999; %outputted time steps
t=logspace(-3,log10(500*24*3600),n);t=t'; %time being investigated (s)
AbsTol=1e-6;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23s(@Eff_mass_acc_model_F,t,y0,options); %solver

%-------Outputs-------
for i=1:6
    Output_Tt(:,i)=Y(:,i)/p_i*1e7; %Film thickness due to SVOC group i (nm)
end

Total_thickness(MM,1)=sum(Output_Tt(n,1:6))+To*1e7; %Total thickness (nm)

end

%----Figure----
figure(1)
semilogx(Diff_input,Total_thickness(:,1))
xlabel('Bulk diffusion coefficient (cm2 s-1)')
ylabel('Film thickness after 500 days (nm)')